// A quick prototype to get RSSI values into the report.
function guruRssi(instanceData) {
    const no_signal_label = "(no signal)";

    function isValidBSSID(bssid) {
        return (
            bssid !== null &&
            bssid !== "00:00:00:00:00:00" &&
            bssid !== "02:00:00:00:00:00"
        );
    }
 
    function format_ssid_label(ssid, bssid) {
        if (ssid === null) {
            return null;
        }

        if (ssid === "") {
            return no_signal_label;
        }

        let out = ssid;

        if (isValidBSSID(bssid)) {
            out += "<br />" + bssid;
        }
        return out;
    }
    function buildSSIDTitle(hasValidBSSID, hasFreqBandInfo) {
        let title = "SSID";
        if (hasFreqBandInfo) {
            title += " [Band]";
        }
        if (hasValidBSSID) {
            title += "/BSSID";
        }
        return title;
    }

    // Utility function to render a textual warning on a chart.  It also links the warning to a web page.
    function render_warning(chart, message, pos_y) {
        let warning = chart.renderer
            .text("Warning: " + message, 80, pos_y)
            .attr({
                zIndex: 5,
            })
            .css({
                fontSize: "12px",
                color: "#F7941C",
                cursor: "pointer",
            })
            .on("click", function () {
                window.open(
                    "https://support.excentis.com/index.php?/Knowledgebase/Article/View/available-wifi-statistics",
                );
            })
            .add();

        return warning;
    }

    let data_series = instanceData.series;
    let unique_ssid_bssids = new Set();
    let ssid_bssid_values = [];
    let rssi_values = [];
    let ssid_bssid_categories = [];
    let ssid_value_value_for_category = {};
    let bssid_value_value_for_category = {};

    var has_valid_rssi = false;
    var min_rssi = 0;

    // Pass 1: collect the "categories" for the SSID/BSSID labels
    //
    // This will collect all required labels for the SSID and BSSID combinations.
    // These will be shown in the
    // - yAxis
    // - tool tip for this series.
    // The collected categories are sorted so this (hopefully) more easy to navigate
    // for the user.
    //
    unique_ssid_bssids.add(no_signal_label);
    for (let ctr = 0; ctr < data_series.length; ++ctr) {
        let sample = data_series[ctr];
        let ssid_bssid_sample = format_ssid_label(sample.ssid, sample.bssid);
        if (ssid_bssid_sample == null) {
            continue;
        }
        unique_ssid_bssids.add(ssid_bssid_sample);
        ssid_value_value_for_category[ssid_bssid_sample] = sample.ssid;
        bssid_value_value_for_category[ssid_bssid_sample] = sample.bssid;
    }

    unique_ssid_bssids.forEach(function (value) {
        ssid_bssid_categories.push(value);
    });
    ssid_bssid_categories.sort(function (a, b) {
        if (a === "(no signal)") {
            return -1;
        }
        if (b === "(no signal)") {
            return 1;
        }

        if (a < b) {
            return 1;
        }

        if (a > b) {
            return -1;
        }

        return 0;
    });

    // pass 2: collect the actual data
    let prevSSID = null;

    // For proper titles/legend
    let hasValidBSSID = false;
    let hasFreqBandInfo = false;

    for (let ctr = 0; ctr < data_series.length; ++ctr) {
        let sample = data_series[ctr];

        // Process the RSSI
        // if the RSSI is a non-null value and
        // - larger or equal than -127, draw the RSSI value.
        // - just larger but not equal than -127, mark is as valid

        let rssi_value = null;
        if (sample.rssi !== null && sample.rssi >= -127) {
            rssi_value = sample.rssi;
        }

        if (rssi_value !== null && rssi_value > -127) {
            has_valid_rssi = true;
        }

        if (rssi_value < min_rssi) {
            min_rssi = rssi_value;
        }

        rssi_values.push([sample.time, rssi_value]);

        // Process the SSID and BSSID
        // 1.  Find the label index in the axis' categories
        // 2.  If the new index is the same as the previous sample, draw the point.
        // 3.  if it not the same, override the index to a null value, so Highcharts doesn't draw a point.
        //     This will cause Highcharts not to draw the vertical lines between 2 horizontal SSID/BSSID
        //     lines.
        //
        let ssid_bssid_sample = format_ssid_label(sample.ssid, sample.bssid);
        let ssid_bssid_idx = null;
        if (ssid_bssid_sample !== "") {
            ssid_bssid_idx = ssid_bssid_categories.indexOf(ssid_bssid_sample);
        }
        if (ssid_bssid_idx !== prevSSID) {
            // insert a dummy timepoint, so we don't draw the vertical lines when the device switches between
            // access points.
            ssid_bssid_values.push([sample.time - 0.001, null]);
            prevSSID = ssid_bssid_idx;
        }
        ssid_bssid_values.push([sample.time, ssid_bssid_idx]);
        hasValidBSSID = hasValidBSSID || isValidBSSID(sample.bssid);
        hasFreqBandInfo =
            hasFreqBandInfo || /\[[0-9.] GHz\]$/.test(sample.ssid);
    }

    // no valid RSSI found?  no series available.
    if (!has_valid_rssi) {
        rssi_values = [];
    }

    let dev_name = data_series[0].name;

    const ssid_bssid_title = buildSSIDTitle(hasValidBSSID, hasFreqBandInfo);
    let chart_settings = {
        title: {
            text: dev_name,
        },
        subtitle: false,
        chart: {
            width: 950,
            zoomType: "x",
            panning: true,
            panKey: "shift",
        },
        xAxis: {
            title: {
                text: "Time [h:min:s]",
            },
            labels: {
                formatter: function () {
                    milliseconds = this.value % 1000;
                    if (milliseconds === 0) {
                        let seconds_part =
                            (this.value - milliseconds) % (1000 * 60);
                        let seconds = seconds_part / 1000;

                        let minutes_part =
                            (this.value - (seconds_part + milliseconds)) %
                            (1000 * 60 * 60);
                        let minutes = minutes_part / (60 * 1000);

                        let hours_part =
                            this.value -
                            (minutes_part + seconds_part + milliseconds);
                        let hours = hours_part / (3600 * 1000);

                        return (
                            util.strMinDigits(hours, 2) +
                            ":" +
                            util.strMinDigits(minutes, 2) +
                            ":" +
                            util.strMinDigits(seconds, 2)
                        );
                    }
                },
            },
            min: 0, // Set the minimum value to 0
            startOnTick: true, // Start the axis on a tick to ensure zero is visible
            max: parseInt(instanceData.parameter_max_time),
        },
        yAxis: [
            {
                id: 0,
                title: {
                    text: "RSSI [dBm]",
                    style: {
                        color: util.excentisblue,
                    },
                },
            },
            {
                id: 1,
                categories: ssid_bssid_categories,
                color: util.excentisgreen,
                min: 0,
                gridLineWidth: 0,
                opposite: true,
                title: {
                    text: ssid_bssid_title,
                    style: {
                        color: util.excentisgreen,
                    },
                },
            },
        ],
        series: [
            {
                animation: false,
                name: "RSSI",
                color: util.excentisblue,
                data: rssi_values,
                zIndex: 1,

                tooltip: {
                    headerFormat: "",
                    pointFormatter: function () {
                        let text = "";
                        let y_value = this.y;
                        if (y_value === 0.0000001) {
                            // workaround to display logarithmic zero value
                            y_value = 0;
                        }
                        text +=
                            '<span style="color:' +
                            this.series.color +
                            '">\u25CF</span> ';
                        text += "RSSI: " + y_value + " dBm <br />";
                        text += "@" + util.strToTime(this.x);
                        return text;
                    },
                },
            },
            {
                name: ssid_bssid_title,
                color: util.excentisgreen,
                data: ssid_bssid_values,
                yAxis: 1,
                zIndex: 0,

                tooltip: {
                    headerFormat: "",
                    pointFormatter: function () {
                        var text = "";
                        let y_value = this.y;
                        let category_value = ssid_bssid_categories[y_value];
                        let ssid_value =
                            ssid_value_value_for_category[category_value];
                        let bssid_value =
                            bssid_value_value_for_category[category_value];
                        if (y_value === 0.0000001) {
                            // workaround to display logarithmic zero value
                            y_value = 0;
                        }
                        text +=
                            '<span style="color:' +
                            this.series.color +
                            '">\u25CF</span><br />';

                        text += "SSID: " + ssid_value + "<br />";
                        text +=
                            "BSSID: " +
                            (isValidBSSID(bssid_value) ? bssid_value : "") +
                            "<br />";
                        text += "@" + util.strToTime(this.x);
                        return text;
                    },
                },
            },
        ],
    };

    // Check conditions and log warnings
    if (!has_valid_rssi) {
        console.warn(
            "RSSI values are not collected. Are you running iOS or iPadOS?",
        );
    }

    if (min_rssi <= -127) {
        const rssiAxis = chart_settings.yAxis[0];
        rssiAxis.min = -130;
        rssiAxis.max = 0;
    }

    return chart_settings;
}
