let CCDFObject;
CCDFObject = CCDFObject || {};

CCDFObject.warnings = [];
CCDFObject.rendered_warnings = [];
CCDFObject.warning_less_than_50_datapoints = "less than 50 data points";
CCDFObject.warning_less_than_10000_packets =
    "less than 10.000 received packets";
CCDFObject.warning_packets_below_range = "packets below range";
CCDFObject.warning_packets_above_range = "packets above range";
CCDFObject.warning_packets_were_sampled = "sampled packets";

CCDFObject.warningsRendered = false;

CCDFObject.updateChartWarnings = function (chart) {
    let show_warning = {};
    show_warning[CCDFObject.warning_less_than_50_datapoints] = false;
    show_warning[CCDFObject.warning_less_than_10000_packets] = false;
    show_warning[CCDFObject.warning_packets_below_range] = false;
    show_warning[CCDFObject.warning_packets_above_range] = false;
    show_warning[CCDFObject.warning_packets_were_sampled] = false;

    for (let idx = 0; idx < CCDFObject.warnings.length; idx++) {
        const message = CCDFObject.warnings[idx][1];
        show_warning[message] |= CCDFObject.warnings[idx][2];
    }

    for (const message in show_warning) {
        // Add message to CCDFObject.rendered_warnings if not yet in there
        if (
            show_warning[message] &&
            !CCDFObject.rendered_warnings.includes(message)
        ) {
            CCDFObject.rendered_warnings.push(message);
        }
    }
};

//This method is responsible for drawing the Latency CDF and CCDF graph.
//Complementary Cumulative Distribution Function (CCDF)
function guruccdf(instanceData) {
    // CCDFObject.warnings = [];
    // A Color picker. Call the picker to get new colors.
    let color_picker = (function () {
        // Using a fixed list of colors. Earlier experience with
        // random RGB colors didn't result in decent separation.
        let colors = [
            util.excentisblue,
            util.excentisorange,
            util.excentisgreen,
            util.excentisred,
            util.excentispurple,
        ];
        let current_idx = -1;
        return function () {
            current_idx = (current_idx + 1) % colors.length;
            return colors[current_idx];
        };
    })();

    function above_zero(val) {
        if (val <= 0) {
            return 1e-7;
        } else {
            return val;
        }
    }

    function toggleAllSeriesWarnings(chart) {
        for (let idx = 0; idx < chart.series.length; idx++) {
            let series = chart.series[idx];

            for (let idx = 0; idx < CCDFObject.warnings.length; idx++) {
                let name = CCDFObject.warnings[idx][0];
                if (series.name === name) {
                    CCDFObject.warnings[idx][2] = !series.visible;
                }
            }
        }

        CCDFObject.updateChartWarnings(chart);
    }

    function toggleSeriesWarnings(series) {
        for (let idx = 0; idx < CCDFObject.warnings.length; idx++) {
            let name = CCDFObject.warnings[idx][0];
            if (series.name === name) {
                CCDFObject.warnings[idx][2] = !series.visible;
            }
        }

        CCDFObject.updateChartWarnings(series.chart);
    }

    // This function is called for each individual chart. The instancedata contains the data for this chart.
    //		A - x_value
    //		B - y_value
    //		C - flow_name
    //		D - height
    //		E - destination_name
    //		F - tick_label

    let latency_conversion = instanceData.latency_conversion;

    let chart_settings = {
        title: {
            text: null,
        },
        chart: {
            height: 450,
            width: 950,
            animation: false,
            backgroundColor: "#ffffff",
            zoomType: "x",
            panning: true,
            panKey: "shift",
            type: "line",
            events: {
                load: function () {
                    let chart = this;
                    let legend = chart.legend;
                    let move_up = 5;
                    let pos_x = legend.box.parentGroup.translateX;
                    let pos_y = legend.box.parentGroup.translateY - move_up;

                    let button = chart.renderer.button(
                        "Invert selection",
                        pos_x,
                        pos_y,
                        function () {
                            toggleAllSeriesWarnings(chart);
                            Highcharts.each(chart.series, function (p) {
                                if (p.visible) {
                                    p.setVisible(false, false);
                                } else {
                                    p.setVisible(true, false);
                                }
                            });
                            chart.redraw();
                        },
                        {
                            padding: 3,
                            width: 100,
                            "text-align": "center",
                        },
                    );

                    button.add();

                    let button_width = button.box.parentGroup.width;
                    let title = legend.title;
                    title.translate(button_width + 3, -move_up);
                },
                render: function () {
                    CCDFObject.updateChartWarnings(this);
                },
            },
        },
        tooltip: {
            useHTML: true,
            headerFormat: null,
            pointFormatter: function () {
                let y_value = this.y;
                if (y_value === 0.0000001) {
                    // workaround to display logarithmic zero value
                    y_value = 0;
                }

                let ns_value = this.category.toFixed(3);
                let p_value = (100 - y_value).toFixed(3);
                let tooltiptext =
                    "<table><tr><td>Latency:</td><td><b>" +
                    ns_value +
                    " " +
                    instanceData.parameter_time_unit +
                    "</b></td>" +
                    '<tr><td style="color:' +
                    this.series.color +
                    '">' +
                    this.series.name +
                    ":</td><td><b>P" +
                    p_value +
                    "</b></td></tr>";

                tooltiptext += "</table>";
                return tooltiptext;
            },
            valueDecimals: 2,
        },
        xAxis: {
            title: {
                text:
                    "Logarithmic Latency [" +
                    instanceData.parameter_time_unit +
                    "]",
            },
            type: "logarithmic",
        },
        yAxis: [
            {
                title: {
                    text: "Logarithmic CDF",
                },
                opposite: true,
                plotLines: [
                    {
                        color: util.excentisgreen,
                        width: 2,
                        dashStyle: "dash",
                        value: 1, // draws a horizontal plotline at P99
                    },
                ],
                labels: {
                    formatter: function () {
                        let str = "P";
                        str += 100.0 - this.value;
                        return str;
                    },
                    style: {
                        color: util.excentisgreen,
                    },
                },
                type: "logarithmic",
                tickInterval: 1,
                minorTickInterval: 0.1,
                endOnTick: true,
                gridLineWidth: 1,
                max: 100.0,
                min: 0.01,
            },
            {
                title: {
                    text: "Logarithmic CCDF",
                },
                labels: {
                    formatter: function () {
                        return this.value / 100.0;
                    },
                    style: {
                        color: util.excentisgrey,
                    },
                },
                type: "logarithmic",
                tickInterval: 1,
                minorTickInterval: 0.1,
                endOnTick: true,
                gridLineWidth: 1,
                max: 100.0,
                min: 0.01,
            },
        ],
        plotOptions: {
            series: {
                marker: {
                    radius: 1,
                    symbol: "circle",
                },
                events: {
                    legendItemClick: function () {
                        toggleSeriesWarnings(this);
                    },
                },
            },
        },
        legend: {
            borderWidth: 0,
            title: {
                style: {
                    fontWeight: "normal",
                },
                text: "or click on individual plots to show/hide:",
            },
        },
        series: [],
        nanos: [],
        chartWarnings: [],
    };

    // Iterate the plot line data objects
    for (const plot of instanceData.series) {
        renderPlotLine(plot);
    }

    function renderPlotLine(plotData) {
        // The lines directly below merge several chart elements into a single distribution chart.
        // let plotLine = instanceData.series // this will contain an entire plotline. The code is called multiple times while loading. Once for each plot line.
        let plotLine = plotData;
        if (typeof plotLine !== "undefined") {
            // when regenerating the entire graph, an emtpy series will be passed along.
            const flowDestination = plotLine[0].C + " : " + plotLine[0].E;
            let drr = instanceData.parameter_data_rate_unit;
            let heights = [];
            let length = plotLine.length;
            let distribution_map_nanos = [];

            // Four extra "fake" objects were added. One at the beginning and three at the end,
            // to pass on additional data. (See GenerateLatencyCcdfCharts to understand)
            let nof_packets_below_min = plotLine[0].B; //value_y_value see GenerateLatencyCcdfCharts why
            let total_nof_sampled_packets = Math.max(1, plotLine[length - 3].B); //value_y_value see GenerateLatencyCcdfCharts why
            let nof_packets_above_max = plotLine[length - 2].B; //value_y_value see GenerateLatencyCcdfCharts why
            let total_nof_rx_packets = plotLine[length - 1].B; //value_y_value see GenerateLatencyCcdfCharts why
            // and skip the last three objects from now on:
            length -= 3;

            let sampled_total =
                nof_packets_below_min +
                total_nof_sampled_packets +
                nof_packets_above_max;

            // start counting at the nof missed packets below range:
            let total_counted = nof_packets_below_min;
            let used_buckets = 0;

            for (
                let idx = 0;
                idx < length && total_nof_sampled_packets > 0;
                idx++
            ) {
                let element = plotLine[idx];
                const value_x_nanos = element.A;
                const value_y_value = element.B;

                if (idx === 0) {
                    // reading the nof packets below min
                    continue; // skip the nof packets below min
                }

                if (value_x_nanos !== null && value_y_value !== null) {
                    total_counted += value_y_value;
                    if (idx > 0 && value_y_value > 0) {
                        used_buckets++;
                    }
                    //let center_of_bin = value_x_nanos + bin_width / 2.
                    let normalized_y_value =
                        100 - (100 * total_counted) / sampled_total;

                    // --- workaround to make the last value visible on log scale:
                    const safe_x_value = above_zero(value_x_nanos);
                    const safe_y_value = above_zero(normalized_y_value);

                    distribution_map_nanos.push([safe_x_value, safe_y_value]);
                }
            }

            let serie = {
                animation: false,
                name: flowDestination,
                nanos: distribution_map_nanos, // nanos is not a highcharts field, we added it so we can dynamically convert the latencies into the desired unit and put it in the data below:
                data: [], // the actual data, in the desired unit, will be filled in below
                color: color_picker(),
                showInLegend: true,
            };

            chart_settings["series"].push(serie);

            // Add warning:
            if (used_buckets < 50) {
                // --- There are 1000 buckets. If less than 50 were filled, maybe the user should use a different range in the Latency Project Properties ?
                CCDFObject.warnings.push([
                    flowDestination,
                    CCDFObject.warning_less_than_50_datapoints,
                    true,
                ]);
            }
            if (total_nof_sampled_packets < 10000) {
                // --- We decided on 22/4/2021 that 10.000 packets is a minimum to get a decent graph
                CCDFObject.warnings.push([
                    flowDestination,
                    CCDFObject.warning_less_than_10000_packets,
                    true,
                ]);
            }
            if (nof_packets_below_min > 0) {
                CCDFObject.warnings.push([
                    flowDestination,
                    CCDFObject.warning_packets_below_range,
                    true,
                ]);
            }
            if (nof_packets_above_max > 0) {
                CCDFObject.warnings.push([
                    flowDestination,
                    CCDFObject.warning_packets_above_range,
                    true,
                ]);
            }
            let expected_nof_samples =
                total_nof_rx_packets -
                nof_packets_below_min -
                nof_packets_above_max;
            if (total_nof_sampled_packets < expected_nof_samples) {
                CCDFObject.warnings.push([
                    flowDestination,
                    CCDFObject.warning_packets_were_sampled,
                    true,
                ]);
            }
        } else {
            // Rerendering the chart, possibly with different latency unit:
            xAxis = chart_settings["xAxis"];
            xAxis.title.text =
                '<span style="color: #F7941C; font-size: 12px; line-height: 1.4640625; font-weight: bold;">Logarithmic Latency [' +
                instanceData.parameter_time_unit +
                "]</span><br>";
        }
    }

    // Convert all latencies from nanos into the desired unit:
    let nof_plots = chart_settings["series"].length;
    for (let idx = 0; idx < nof_plots; idx++) {
        let serie = chart_settings["series"][idx];
        let nof_dots = serie.nanos.length;
        let converted_dots = [];
        for (let dotid = 0; dotid < nof_dots; dotid++) {
            dot = serie.nanos[dotid];
            let x_nanos = dot[0];
            let y_value = dot[1];
            let x_in_desired_unit = latency_conversion(x_nanos);
            converted_dots.push([x_in_desired_unit, y_value]);
        }

        serie.data = converted_dots;
    }

    return chart_settings;
}
