document.addEventListener("DOMContentLoaded", () => {
    // Wait for the Highcharts graphs to be loaded
    setTimeout(async function () {
        const actionTables = document.querySelectorAll('.action-table');

        actionTables.forEach(table => {
            new ActionTable(table);
        });

        // Select all rows except the ones in the favorites section.
        // CollapsibleRow object for favorited rows is initialized in the function 'updateFavorites'.
        const collapseRows = document.querySelectorAll('.collapse-row:not(#favorites .collapse-row)');
        // Initialize collapsible rows
        collapseRows.forEach(row => {
            // Change the bool passed to false, to expand the rows in Results by default
            new CollapsibleRow(row, true);
        });

        // Initialize collapsible sections
        const collapsibleSections = document.querySelectorAll('.collapse');
        collapsibleSections.forEach(section => new CollapsibleSection(section));

        loadPage();
        dragFunctionality();

    }, 200);
});

// Functionality for the drag and drop table rows
function dragFunctionality() {
    const containers = document.querySelectorAll('.container');

    containers.forEach(container => {

        const sortableTable = dragula([container], {
            moves: function (el, container, handle) {
                return handle.classList.contains('tr-drag-icon');
            }
        });

        let draggedRow = null;
        let draggedRowChild = null;

        sortableTable.on('drag', function (row) {
            draggedRow = row;
            if (row.classList.contains('collapse-row')) {
                draggedRowChild = row.nextElementSibling;
            }
        });

        sortableTable.on('dragend', function () {
            let prevRow = draggedRow.previousElementSibling;
            let nextRow = draggedRow.nextElementSibling;

            // Prevent a row to be dragged between a collapse row and its content row
            if (prevRow !== null && (prevRow.classList.contains('collapse-row'))) {
                nextRow.insertAdjacentElement("afterend", draggedRow);
            }

            // If the row being dragged is a collapsible row, move the content row as well
            if (draggedRow && draggedRowChild) {
                draggedRow.insertAdjacentElement("afterend", draggedRowChild);
            }

            draggedRow = null;
            draggedRowChild = null;
        });
    });
}

class CollapsibleSection {
    constructor(element) {
        this.element = element;
        this.collapseContent = element.querySelector(".collapse-content");
        this.btnCollapse = iconFactory.createCollapseIcon();
        this.btnCollapsePolyline = this.btnCollapse.querySelector('.btn-collapse polyline');
        this.collapsed = false;
        this.height = this.element.offsetHeight;
        this.element.style.height = "auto";

        if (this.element.id !== 'quote') {
            this.addCollapseIconToHeader();
        } else {
            this.addCollapseIconToQuote();
        }
    }

    addCollapseIconToHeader() {
        let contentHeader = this.element.querySelector('.content-header');
        contentHeader.appendChild(this.btnCollapse);
        this.btnCollapse.addEventListener('click', this.toggle.bind(this));
    }

    addCollapseIconToQuote() {
        this.element.appendChild(this.btnCollapse);
        this.btnCollapse.addEventListener('click', this.toggle.bind(this));
    }

    toggle() {
        if (!this.collapsed) {
            this.btnCollapsePolyline.style.transform = "rotate(-180deg)";
            this.collapseContent.style.opacity = "0";
            setTimeout(() => {
                this.element.style.height = "75px";
                this.collapseContent.style.display = "none";
            }, 150);
        } else {
            this.element.style.height = "auto";
            this.collapseContent.style.display = "block";
            this.btnCollapsePolyline.style.transform = "rotate(0deg)";
            setTimeout(() => {
                this.collapseContent.style.opacity = "1";
            }, 150);
        }

        this.collapsed = !this.collapsed;
    }
}

// TODO: inherit from CollapsibleSection
class CollapsibleRow {
    constructor(element, collapsed=true) {
        this.element = element;
        this.nextRow = this.element.nextElementSibling; // this is the collapsible content
        this.btnCollapsePolyline = this.element.querySelector('.btn-collapse polyline');
        this.btnCollapse = this.element.querySelector('.btn-collapse');

        this.btnCollapse.addEventListener('click', this.toggle.bind(this));

        // Make sure clicking the row triggers collapse/expand as well
        this.element.addEventListener('click', (event) => {
            /*
             Make sure clicking the star icon does not trigger collapse toggle
             and clicking the collapse button does not trigger the toggle as well
             cause then it would be toggled twice.
             */
            if (!event.target.closest('.icon-star') && !event.target.closest('.btn-collapse')) {
                this.toggle();
            }
        });

        this.initialState(collapsed);


    }

    initialState(collapsed) {
        // this.element.setAttribute('data-collapsed', collapsed');
        if (collapsed === true) {
            this.btnCollapsePolyline.style.transform = "rotate(-180deg)";
            this.nextRow.style.opacity = "0";
            setTimeout(() => {
                this.nextRow.style.display = "none";
            }, 150);
            this.element.setAttribute('data-collapsed', 'true');
            this.element.classList.remove('expanded');
        } else {
            this.btnCollapsePolyline.style.transform = "rotate(0deg)";
            this.nextRow.style.display = "table-row";
            setTimeout(() => {
                this.nextRow.style.opacity = "1";
            }, 0);
            this.element.setAttribute('data-collapsed', 'false');
            this.element.classList.add('expanded');
        }
    }

    toggle() {
        let collapsed = this.element.getAttribute('data-collapsed');
        if (collapsed === 'false') {
            this.btnCollapsePolyline.style.transform = "rotate(-180deg)";
            this.nextRow.style.opacity = "0";
            setTimeout(() => {
                this.nextRow.style.display = "none";
            }, 150);
            this.element.setAttribute('data-collapsed', 'true');
            this.element.classList.remove('expanded');
        } else {
            this.btnCollapsePolyline.style.transform = "rotate(0deg)";
            this.nextRow.style.display = "table-row";
            setTimeout(() => {
                this.nextRow.style.opacity = "1";
            }, 0);
            this.element.setAttribute('data-collapsed', 'false');
            this.element.classList.add('expanded');
        }
    }
}

class Row {
    constructor(element, parent, rowId) {
        this.element = element;
        this.rowId = rowId;
        this.parent = parent;
        this.parentTableId = this.parent.tableId;
        this.isCollapseRow = this.element.classList.contains('collapse-row');

        this.starIcon = iconFactory.createStarIcon();
        this.eyeOpenIcon = iconFactory.createEyeOpenIcon();
        this.eyeClosedIcon = iconFactory.createEyeClosedIcon();
        this.btnCollapse = iconFactory.createCollapseIcon();
        this.btnDrag = iconFactory.createDragIcon();

        // Bind events to icons
        this.starIcon.onclick = this.favoriteRowToggle.bind(this);
        this.eyeOpenIcon.onclick = this.visibilityToggle.bind(this);
        this.eyeClosedIcon.onclick = this.visibilityToggle.bind(this);

        // Add icons to the row
        this.addIconsToRow();

        // Set the data attributes
        this.element.setAttribute("data-row-id", rowId);
        this.element.setAttribute("data-visible", "true");
        this.element.setAttribute("data-favorite", "false");
    }

    addIconsToRow() {
        // Create td for drag icon
        let tdDrag = document.createElement('td');
        tdDrag.classList.add('tr-drag-row');
        tdDrag.append(this.btnDrag);

        // Prepend td to the table body
        this.element.prepend(tdDrag);

        // Create td for buttons and button container
        let tdButtons = document.createElement('td');
        tdButtons.classList.add('button-td');
        let buttonContainer = document.createElement('div');
        buttonContainer.classList.add('td-button-container');

        // Star
        buttonContainer.append(this.starIcon);

        // Visibility icons
        // buttonContainer.append(this.eyeOpenIcon);
        // buttonContainer.append(this.eyeClosedIcon);

        // Collapse button
        if (! this.isCollapseRow) {
            this.btnCollapse.classList.add('btn-collapse-inactive');
        }

        buttonContainer.append(this.btnCollapse);
        tdButtons.appendChild(buttonContainer);
        this.element.appendChild(tdButtons);
    }

    visibilityToggle() {

        if (this.element.getAttribute('data-visible') === "true") {
            // Hide the row
            this.element.setAttribute('data-visible', "false");
            this.eyeOpenIcon.style.display = "none";
            this.eyeClosedIcon.style.display = "block";
        } else {
            // Show the row
            this.element.setAttribute('data-visible', "true");
            this.eyeOpenIcon.style.display = "block";
            this.eyeClosedIcon.style.display = "none";
        }

        this.showHideRows();
    }

    removeRowFromFavoriteData() {
        // Remove row id from favoriteData for corresponding parent table id
        let rowIndex = favoriteData[report_id][this.parentTableId].indexOf(this.rowId);
        if (rowIndex > -1) {
            favoriteData[report_id][this.parentTableId].splice(rowIndex, 1);
        }

        // Update local storage
        localStorage.setItem('favoriteData', JSON.stringify(favoriteData));
    }

    // When a row is favorited, the table id and row id are stored in favoriteData
    favoriteRowToggle() {
        if (this.element.getAttribute('data-favorite') === "true") {
            // Remove row id from favoriteData for corresponding parent table id
            this.removeRowFromFavoriteData();

            // update visual
            this.starIcon.classList.remove('icon-star-selected');
            this.starIcon.classList.add('icon-star-deselected');
            this.starIcon.classList.remove('hide');
        } else {
            // Store the table id in favoriteData is not present yet
            favoriteData[report_id][this.parentTableId] = favoriteData[report_id][this.parentTableId] || [];
            // Add row id to favoriteData for corresponding parent table id
            favoriteData[report_id][this.parentTableId].push(this.rowId);

            // update visual
            this.starIcon.classList.remove('icon-star-deselected');
            this.starIcon.classList.add('icon-star-selected');
            this.starIcon.classList.add('hide');
        }

        this.parent.updateFavorites();

        // Update local storage
        localStorage.setItem('favoriteData', JSON.stringify(favoriteData));
    }

    // FIXME: code duplication
    updateStarFill() {
        if (this.element.getAttribute('data-favorite') === "true") {
            this.starIcon.classList.remove('icon-star-deselected');
            this.starIcon.classList.add('icon-star-selected');
            this.starIcon.classList.remove('hide');
        } else {
            this.starIcon.classList.remove('icon-star-selected');
            this.starIcon.classList.add('icon-star-deselected');
            this.starIcon.classList.add('hide');
        }
    }

    showHideRows() {
        const hide = this.parent.element.getAttribute('data-hide-rows')
        if (hide === 'true' && this.element.getAttribute('data-visible') === "false") {
            this.element.style.display = "none";
            if (this.isCollapseRow) {
                this.getCollapsibleContentRow().style.display = "none";
            }
        } else if (hide === 'false' && this.element.getAttribute('data-visible') === "false") {
            this.element.style.display = "table-row";
            if (this.isCollapseRow && this.element.getAttribute('data-collapsed') === "false") {
                this.getCollapsibleContentRow().style.display = "table-row";
            }
        }
    }

    getCollapsibleContentRow() {
        return this.element.nextElementSibling;
    }
}

class FavoriteRow extends Row {
    constructor(element, parentTable, rowId) {
        super(element, parentTable, rowId);

        // set data attributes
        this.element.setAttribute("data-favorite", "true");

        // update visual
        this.starIcon.classList.remove('icon-star-deselected');
        this.starIcon.classList.add('icon-star-selected');

        this.collapseContent = null;

        // Bind click event to the throughput unit selectors
        this.throughputUnitSelectors = document.querySelectorAll('.throughput-units li');
        this.throughputUnitSelectors.forEach((selector) => {
            selector.addEventListener('click', (event) => {
                this.reRenderGraph();
            });
        });

        // Bind click event to the latency unit selectors
        this.latencyUnitSelectors = document.querySelectorAll('.latency-units li');
        this.latencyUnitSelectors.forEach((selector) => {
            selector.addEventListener('click', (event) => {
                this.reRenderGraph();
            });
        });

        this.lazyLoadCallback = this.lazyLoadCallback.bind(this);

        // Intersection Observer configuration
        this.observer = new IntersectionObserver(this.lazyLoadCallback, {
            root: null, // Use the viewport as the root
            rootMargin: '0px',
            threshold: 0.2, // Trigger when 50% of the chart is visible
        });

        // Flag to track whether the chart has been adjusted
        this.chartSettingsChanged = false;

        // Render graphs in favorites on print if they are not rendered yet through lazy loading
        window.onbeforeprint = (event) => {
            this.createChart();
        };
    }

    createChart() {
        this.chartCreated = true;
        if (this.collapseContent !== null) {
            const content = this.collapseContent.querySelector('.inner-table-content');
            let graphContainers = content.querySelectorAll('.graph-container');
            if (graphContainers !== null) {
                graphContainers.forEach(graphContainer => {
                    // Get the id of the class for graph
                    const graphClass = Array.from(graphContainer.classList).find(className => className !== 'graph-container');

                    // Remove the script section
                    const script = content.querySelector('script');
                    if (script !== null) {
                        content.querySelector('script').remove();
                    }

                    // Look if there are any existing graphs and if so remove them
                    graphContainer.querySelectorAll('.graph').forEach(graph => {
                        graph.remove();
                    });

                    // Create a new div for the updated graph
                    let graph = document.createElement('div');
                    graph.classList.add('graph');

                    if (graphClass) {
                        graphContainer.appendChild(graph);
                        Highcharts.chart(graph, chartSettings[graphClass]);
                        this.observer.unobserve(graph);

                    } else {
                        console.error('No class found for graph');
                    }
                });
            }

            this.chartSettingsChanged = false;
        }
    }

    lazyLoadCallback(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting && this.chartSettingsChanged) {
                // If the chart is visible, render it and disconnect the observer
                this.createChart();
            }
        });
    }

    reRenderGraph() {

        if (this.collapseContent !== null) {

            // Used to determine whether to re-render the chart
            this.chartSettingsChanged = true;
            const content = this.collapseContent.querySelector('.inner-table-content');
            const graphContainers = content.querySelectorAll('.graph-container');

            if (graphContainers !== null) {
                graphContainers.forEach(graphContainer => {
                    let graph = document.createElement("div");
                    graph.classList.add("graph");
                    graphContainer.appendChild(graph);
                    this.observer.observe(graphContainer);
                });

            }
        }
    }

    favoriteRowToggle() {
        this.removeRowFromFavoriteData();
        // Remove the row from the table
        this.element.remove();
        this.parent.updateFavorites();
    }

    setCollapseContent(row) {
        this.collapseContent = row;
        this.collapseContent.setAttribute('data-row-id', this.rowId);
        this.reRenderGraph();
    }

    getCollapsibleContentRow() {
        return this.collapseContent;
    }
}

// Used for assigning a unique id to each table
let table_id_counter = 0;
// Shared data source to store favorited rows. Parse data from cookie if present else initialize empty object
let favoriteData = localStorage.getItem('favoriteData') ? JSON.parse(localStorage.getItem('favoriteData')) : {};
favoriteData[report_id] ??= {};

const iconFactory = new IconFactory();

class ActionTable {
    constructor(element) {
        this.element = element;
        this.rows = [];
        this.rowId = 0;
        this.tableId = table_id_counter++;
        this.element.setAttribute("data-hide-rows", "false");
        this.element.setAttribute("data-table-id", this.tableId);

        // Create icons using factory
        this.eyeClosedIcon = iconFactory.createEyeClosedIcon();
        this.eyeOpenIcon = iconFactory.createEyeOpenIcon();

        // Add event listeners to the icons
        this.eyeOpenIcon.onclick = this.toggleVisibility;
        this.eyeClosedIcon.onclick = this.toggleVisibility;

        this.initializeRows();
        this.initializeFavoritesTable();

        // Call updateFavorites to update favorite rows based on cookie data if present
        if (favoriteData[report_id][this.tableId]) {
            this.updateFavorites();
        }
    }

    initializeRows() {
        // Add eye icon to table header
        const tableHeader = this.element.querySelector('thead tr');

        // Add th to the table header to hold to icon
        // const th = document.createElement('th');
        // th.classList.add('btn-visibility-toggle');

        // Add icon to the th
        // th.appendChild(this.eyeOpenIcon);
        // th.appendChild(this.eyeClosedIcon);

        // tableHeader.append(th);

        // TODO: differentiate between collapsible row and regular row
        const rows = this.element.querySelectorAll('tbody tr:not(.collapse-row-content)');
        rows.forEach(row => {
            let tableRow = new Row(row, this, this.rowId++);
            this.rows.push(tableRow);
        });
    }

    initializeFavoritesTable() {
        // Create the favorites table with same header and table id corresponding to this table id
        const favoritesTable = this.element.cloneNode(true);

        // Get content header
        const header = this.element.parentElement.parentElement.querySelector('h2').innerHTML;
        // Select sub header if it exists
        const table = this.element.parentElement;
        const subHeaderElement = table.closest('h3');
        const subHeader = subHeaderElement ? subHeaderElement.innerHTML : "";

        // Clear the body of the favorites table
        favoritesTable.querySelector('tbody').innerHTML = "";

        // Remove the eye icon
        // favoritesTable.querySelector('.btn-visibility-toggle').remove();

        // Wrap the favorites table in a div with classes collapse and content
        const favoritesTableWrapper = document.createElement('div');
        favoritesTableWrapper.classList.add('collapse');
        favoritesTableWrapper.classList.add('content');
        favoritesTableWrapper.classList.add('favorites-table');

        // add content header
        const favoritesTableHeader = document.createElement('div');
        favoritesTableHeader.classList.add('content-header');
        favoritesTableHeader.innerHTML = "<h2 class=\"favorites-title\">" + header + "</h2>";

        // Add the header to the wrapper
        favoritesTableWrapper.append(favoritesTableHeader);

        // Create another wrapper for collapsible content
        const favoritesTableContent = document.createElement('div');
        favoritesTableContent.classList.add('collapse-content');

        // add the table to the favorites' table content
        favoritesTableContent.append(favoritesTable);

        // Add favoritesTableContent to favoritesTableWrapper
        favoritesTableWrapper.append(favoritesTableContent);

        // add the wrapper to favorites container
        document.querySelector('#favorites-container .collapse-content').append(favoritesTableWrapper);
    }

    updateFavorites = () => {
        const favoriteRows = favoriteData[report_id][this.tableId] || [];

        // Use favoriteData to update the data attributes of the rows
        this.rows.forEach(row => {
            const isFavorite = favoriteRows.includes(row.rowId);
            row.element.setAttribute('data-favorite', isFavorite);
            row.updateStarFill();
        });

        const favoritesTable = document.querySelector(`#favorites-container [data-table-id="${this.tableId}"]`);

        // Make favorites table visible if there are rows favorited. Hide otherwise.
        // FIXME: replace with adding / removing class
        const favoriteRowCount = favoriteData[report_id][this.tableId].length;
        favoritesTable.closest('.favorites-table').style.display = favoriteRowCount > 0 ? "block" : "none";

        // Select the body of corresponding favorites table
        const favoritesTableBody = favoritesTable.querySelector('tbody');

        this.rows.forEach(row => {
            const favorite = row.element.getAttribute('data-favorite');
            if (favorite === "true") {
                // Add a clone of the row to favorites table body if it does not exist
                const favoriteRow = favoritesTableBody.querySelector(`[data-row-id="${row.rowId}"]`);

                // Add a favorite row if it does not exist
                if (favoriteRow === null) {
                    // Clone the row (tr element)
                    let rowClone = row.element.cloneNode(true);
                    rowClone.querySelector('.button-td').remove();
                    rowClone.querySelector('.tr-drag-row').remove();

                    // FIXME: favoriteRowInstance.getCollapsibleContentRow() returns null at some point
                    // Create instance of FavoriteRow
                    const favoriteRowInstance = new FavoriteRow(rowClone, this, row.rowId);
                    favoritesTableBody.append(favoriteRowInstance.element);

                    // Check if the row contains collapsible content
                    if (favoriteRowInstance.isCollapseRow) {

                        // Clone the collapsible content
                        let collapsibleContentClone = row.getCollapsibleContentRow().cloneNode(true)
                        /*
                          Look for graph containers in the collapsible content and clear them.
                          This is done because we can't clone the static Highcharts graph, after the DOM is loaded,
                          without re-rendering / re-executing the script that generates the graph.
                         */
                        collapsibleContentClone.querySelectorAll('.graph-container').forEach(graphContainer => {
                            // Iterate all graphs in graph container
                            const graph = graphContainer.querySelector('.graph')
                            // clear all graph
                            graph.innerHTML = "";
                            graph.remove();

                        });
                        favoriteRowInstance.setCollapseContent(collapsibleContentClone);

                        // TODO: WATCH OUT FOR THIS ONE
                        // Remove graph content from row clone (fix for bug where graph gets duplicated when favoriting row)
                        favoriteRowInstance.element.querySelectorAll('.graph').forEach(graph => {
                            graph.remove();
                        });

                        favoritesTableBody.append(favoriteRowInstance.getCollapsibleContentRow());

                        new CollapsibleRow(favoriteRowInstance.element, false);
                    }
                }
            } else {
                // Remove the row from the favorites table body
                const favoriteRows = favoritesTableBody.querySelectorAll(`[data-row-id="${row.rowId}"]`);

                favoriteRows.forEach(favoriteRow => {
                    if (favoriteRow !== null) {
                        favoriteRow.remove();
                    }
                });
            }
        });

        // if favoritesData has no rows, display the no favorites message
        const message = document.querySelector('#favorites-container .no-rows-message');
        // iterate all keys of favoriteData and see if there are any rows
        let favoriteCount = Object.values(favoriteData[report_id]).reduce((total, arr) => total + arr.length, 0);

        if (favoriteCount === 0) {
            message.style.display = "block";
        } else {
            message.style.display = "none";
        }
    }

    toggleVisibility = () => {
        if (this.element.getAttribute('data-hide-rows') === "true") {
            this.element.setAttribute('data-hide-rows', "false");
            this.eyeOpenIcon.style.display = "block";
            this.eyeClosedIcon.style.display = "none";
        } else {
            this.element.setAttribute('data-hide-rows', "true");
            this.eyeOpenIcon.style.display = "none";
            this.eyeClosedIcon.style.display = "block";
        }

        this.rows.forEach(row => {
            row.showHideRows();
        });
    }
}