let chartSettings = {};

// Global highcharts settings
Highcharts.setOptions({
    credits: false,
    chart: {
        style: {
            fontFamily: "'Open Sans', sans-serif"
        },
    },
    legend: {
        borderColor: util.excentislightgrey,
        borderWidth: 1,
    },
    xAxis: {
        labels: {
            style: {
                color: util.excentisgrey,
            }
        }
    },
    // We don't need more options in the context menu than listed below in menuItems: [...]
    exporting: {
        buttons: {
            contextButton: {
                enabled: true,
                menuItems: ["viewFullscreen", "printChart", "separator", "downloadPNG", "downloadJPEG", "downloadPDF"]
            }
        }
    }
});


class Graph {
    constructor() {
        // Default values
        this.latency_unit = "ms";
        this.throughput_unit = "Mbps";

        // Bind click event to the throughput unit selectors
        this.throughputUnitSelectors = document.querySelectorAll('.throughput-units li');
        this.throughputUnitSelectors.forEach((selector) => {
            selector.addEventListener('click', (event) => {
                this.select_new_throughput(event);
            });
        });

        // Bind click event to the latency unit selectors
        this.latencyUnitSelectors = document.querySelectorAll('.latency-units li');
        this.latencyUnitSelectors.forEach((selector) => {
            selector.addEventListener('click', (event) => {
                this.select_new_latency(event);
            });
        });

        // Bind the render method to the instance to use it as a callback
        this.lazyLoadCallback = this.lazyLoadCallback.bind(this);

        // Intersection Observer configuration
        this.observer = new IntersectionObserver(this.lazyLoadCallback, {
            root: null, // Use the viewport as the root
            rootMargin: '0px',
            threshold: 0.1, // Trigger when 50% of the chart is visible
        });

        // Flag to track whether the chart has been created
        this.chartCreated = false;
        this.chartSettingsChanged = false;
    }

    lazyLoadCallback(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting && this.chartSettingsChanged) {
                // If the chart is visible, render it and disconnect the observer
                this.createChart();
                this.chartCreated = true;
            }
        });
    }

    select_new_throughput(event) {
        // Remove 'selected' class from all elements
        this.throughputUnitSelectors.forEach(selector => {
            selector.classList.remove('selected');
        });

        // Add 'selected' class to the clicked element
        event.currentTarget.classList.add('selected');

        // Get value of the clicked element
        this.throughput_unit = event.currentTarget.textContent;

        // re-render the graph
        this.render();
    }

    select_new_latency(event) {
        // Remove 'selected' class from all elements
        this.latencyUnitSelectors.forEach(selector => {
            selector.classList.remove('selected');
        });

        // Add 'selected' class to the clicked element
        event.currentTarget.classList.add('selected');

        // Get value of the clicked element
        this.latency_unit = event.currentTarget.textContent;

        // re-render the graph
        this.render();
    }

    render() {
        // PLACEHOLDER - TO BE OVERWRITTEN BY IMPLEMENTING CLASSES
    }

    createChart() {
        // PLACEHOLDER - TO BE OVERWRITTEN BY IMPLEMENTING CLASSES
    }
}

class WifiGraph extends Graph {
    constructor(id, values) {
        super();
        this.id = id;
        this.values = values;
    }

    createChart() {
        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            const graph = element.querySelector('.graph');
            // element.innerHTML = "";
            // Create Highcharts chart
            Highcharts.chart(graph, chartSettings[this.id]);

            this.observer.unobserve(element);
        });

        this.chartSettingsChanged = false;
    }

    render() {
        // Used to determine whether to re-render the chart
        this.chartSettingsChanged = true;

        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        const instanceData = {
            "series": this.values,
            "id": this.id,
            "height": 300,
            "parameter_max_time": max_time
        };

        // Parse & update the settings generated by 'guru' to a chartSettings object
        chartSettings[this.id] = guruRssi(instanceData);

        chartRenderTo.forEach(element => {
            addGraphDivToElement(element);
            this.observer.observe(element);
        });
    };
}

class TCPGraph extends Graph {
    constructor(id, values) {
        super();
        this.id = id;
        this.values = values;
    }

    createChart() {
        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            const graph = element.querySelector('.graph');
            // element.innerHTML = "";
            // Create Highcharts chart
            Highcharts.chart(graph, chartSettings[this.id]);

            this.observer.unobserve(element);
        });

        this.chartSettingsChanged = false;
    }

    render() {
        // Used to determine whether to re-render the chart
        this.chartSettingsChanged = true;

        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            element.innerHTML = "";
        });

        const instanceData = {
            "parameter_data_rate_unit": this.throughput_unit,
            "parameter_time_unit": util.get_latency_unit_name(this.latency_unit),
            "throughput_conversion": util.conversion_function(this.throughput_unit),
            "latency_conversion": util.conversion_function(this.latency_unit),
            "series": this.values,
            "id": this.id,
            "height": 300,
            "parameter_max_time": max_time
        };

        // Parse & update the settings generated by 'guru' to a chartSettings object
        chartSettings[this.id] = guru(instanceData);

        chartRenderTo.forEach(element => {
            addGraphDivToElement(element);
            this.observer.observe(element);
        });
    };
}

class AggregatedTCPGraph extends Graph {
    constructor(id, values) {
        super();
        this.id = id;
        this.values = values;
    }

    createChart() {
        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            const graph = element.querySelector('.graph');
            // element.innerHTML = "";
            // Create Highcharts chart
            Highcharts.chart(graph, chartSettings[this.id]);

            this.observer.unobserve(element);
        });

        this.chartSettingsChanged = false;
    }

    render() {
        // Used to determine whether to re-render the chart
        this.chartSettingsChanged = true;

        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        const instanceData = {
            "parameter_time_unit": util.get_latency_unit_name(this.latency_unit),
            "throughput_conversion": util.conversion_function(this.throughput_unit),
            "latency_conversion": util.conversion_function(this.latency_unit),
            "parameter_data_rate_unit": this.throughput_unit,
            "series": this.values,
            "id": this.id,
            "height": 300,
            "parameter_max_time": max_time
        };

        // Update the settings generated by 'guru' to a chartSettings object
        chartSettings[this.id] = guruAggregatedTCP(instanceData);

        chartRenderTo.forEach(element => {
            addGraphDivToElement(element);
            this.observer.observe(element);
        });
    };
}

class FrameBlastingGraph extends Graph {
    constructor(id, values) {
        super();
        this.id = id;
        this.values = values;
    }

    createChart() {
        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            const graph = element.querySelector('.graph');
            // element.innerHTML = "";
            // Create Highcharts chart
            Highcharts.chart(graph, chartSettings[this.id]);

            this.observer.unobserve(element);
        });

        this.chartSettingsChanged = false;
    }

    render() {
        // Used to determine whether to re-render the chart
        this.chartSettingsChanged = true;

        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        const instanceData = {
            "parameter_data_rate_unit": this.throughput_unit,
            "parameter_time_unit": util.get_latency_unit_name(this.latency_unit),
            "throughput_conversion": util.conversion_function(this.throughput_unit),
            "latency_conversion": util.conversion_function(this.latency_unit),
            "series": this.values,
            "id": this.id,
            "height": 300,
            "parameter_max_time": max_time
        };

        // Parse and update the settings generated by 'guru' to a chartSettings object
        chartSettings[this.id] = gurufb(instanceData);

        chartRenderTo.forEach(element => {
            addGraphDivToElement(element);
            this.observer.observe(element);
        });

    };
}

class FrameBlastingDistributionGraph extends Graph {
    constructor(id, values) {
        super();
        this.id = id;
        this.values = values;
        this.guru = null;
    }

    createChart() {
        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            const graph = element.querySelector('.graph');
            // element.innerHTML = "";
            // Create Highcharts chart
            Highcharts.chart(graph, chartSettings[this.id]);

            this.observer.unobserve(element);
        });

        this.chartSettingsChanged = false;
    }

    render() {

        // Used to determine whether to re-render the chart
        this.chartSettingsChanged = true;

        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        // clear content of the div
        chartRenderTo.forEach(element => {
            element.innerHTML = "";
        });

        const instanceData = {
            "parameter_time_unit": util.get_latency_unit_name(this.latency_unit),
            "latency_conversion": util.conversion_function(this.latency_unit),
            "series": this.values,
        }

        // Parse and update the settings generated by 'guru' to a chartSettings object
        chartSettings[this.id] = gurudist(instanceData);

        chartRenderTo.forEach(element => {
            addGraphDivToElement(element);
            this.observer.observe(element);
        });
    }
}

class FrameBlastingLatencyCCDFGraph extends Graph {
    constructor(id, values) {
        super();
        this.id = id;
        this.values = values;
        this.instanceData = null;
    }

    createChart() {
        const chartRenderTo = document.querySelectorAll("." + this.id);

        chartRenderTo.forEach(graphContainer => {
            const graph = graphContainer.querySelector('.graph');
            const settings = chartSettings[this.id];
            Highcharts.chart(graph, settings);
            this.observer.unobserve(graphContainer);
        });

        this.chartSettingsChanged = false;
    }

    /*
    The CCDF graph is a special case, as it requires a call to updateChartWarnings() before the chart is created.
    This is because if you favorite a row containing a CCDF graph, the warnings need to be copied.
    However, if the updateChartWarnings() is only called the moment the graph is created
    (called inside the render function of the Highcharts settings), those warnings don't exist yet.
     */
    render() {
        // Used to determine whether to re-render the chart
        this.chartSettingsChanged = true;

        // Get classes with id
        const chartRenderTo = document.querySelectorAll("." + this.id);

        this.instanceData = {
            "parameter_time_unit": util.get_latency_unit_name(this.latency_unit),
            "latency_conversion": util.conversion_function(this.latency_unit),
            "series": this.values,
            "id": this.id,
        }

        // Parse and update the settings generated by 'guru' to a chartSettings object
        const settings = guruccdf(this.instanceData);
        chartSettings[this.id] = settings;

        // Call updateChartWarnings() method to create the warnings
        CCDFObject.updateChartWarnings(settings);

        chartRenderTo.forEach(element => {
            // Look if there's a warning list already present, if not, create it
            if (!element.querySelector(".graph-warning-list")) {
                // Create text element to display above warning list
                let graphWarningText = document.createElement("p");
                graphWarningText.classList.add("graph-warning-text");
                graphWarningText.innerHTML = "The following warnings were generated while rendering this graph:";

                // Create ul element to hold warnings
                let warningList = document.createElement("ul");
                warningList.classList.add("graph-warning-list");

                for (const warning of CCDFObject.rendered_warnings) {

                    // Create li element to hold warning
                    let warningElement = document.createElement("li");
                    warningElement.innerHTML = warning;
                    warningList.appendChild(warningElement);
                }

                element.appendChild(graphWarningText);
                element.appendChild(warningList);
            }

            addGraphDivToElement(element);
            this.observer.observe(element);
        });
    }
}

function addGraphDivToElement(element) {
    // Create div with class graph_container
    let graphContainer = document.createElement("div");
    graphContainer.classList.add("graph");

    element.appendChild(graphContainer);
}