//This method is responsible for drawing the Latency CDF and CCDF graph.
//Complementary Cumulative Distribution Function (CCDF) 
define('guruccdf', ["require", 'bb_highcharts'], function (require) {
    var Highcharts = require('bb_highcharts');

    // The Excentis colors. 
    let excentis_orange = '#F7941C';
    let excentis_green = '#00A650';
    let excentis_blue = '#00AEEF';
    let excentis_red = '#ED1B23';
    let excentis_magenta = '#EC008C';

    let warning_less_than_50_datapoints = 'less than 50 data points'
    let warning_less_than_10000_packets = 'less than 10.000 received packets'
    let warning_packets_below_range = 'packets below range'
    let warning_packets_above_range = 'packets above range'
    let warning_packets_were_sampled = 'sampled packets'

    // A Color picker. Call the picker to get new colors.
    var color_picker = function () {
        // Using a fixed list of colors. Earlier experience with 
        // random RGB colors didn't result in decent separation.
        let colors = [excentis_blue, excentis_orange, excentis_green, excentis_red, excentis_magenta];
        let current_idx = -1
        return function () {
            current_idx = (current_idx + 1) % colors.length;
            return colors[current_idx];
        }
    }();

    var chart_layer;
    var warnings = []
    var rendered_warnings = []

    function render(chart, message, pos_y) {
        let warning = chart.renderer.text('Warning: flow detected with ' + message, 90, pos_y)
            .attr({
                zIndex: 5
            })
            .css({
                fontSize: '12px',
                color: '#F7941C'
            })
            .on('click', function () {
                window.open("https://support.excentis.com/index.php?/Knowledgebase/Article/View/latency-cdf-ccdf#paragraph " + message);
            })
            .add();
        rendered_warnings.push([message, warning])
        return warning
    }

    function cleanupChartWarnings(chart) {
        for (let idx = 0; idx < rendered_warnings.length; idx++) {
            warning = rendered_warnings[idx][1]
            warning.destroy()
        }
        rendered_warnings = []
    }

    function getText(chart, message) {
        let pos_y_offset = 0
        for (let idx = 0; idx < rendered_warnings.length; idx++) {
            if (rendered_warnings[idx][0] == message) {
                return rendered_warnings[idx][1]
            }
            pos_y_offset += 20
        }

        return render(chart, message, 220 + pos_y_offset)
    }
    function above_zero(val) {
        if (val <= 0) {
            return 1e-7
        } else {
            return val
        }
    }

    function updateChartWarnings(chart) {
        var show_warning = {};
        show_warning[warning_less_than_50_datapoints] = false;
        show_warning[warning_less_than_10000_packets] = false;
        show_warning[warning_packets_below_range] = false;
        show_warning[warning_packets_above_range] = false;
        show_warning[warning_packets_were_sampled] = false;

        for (let idx = 0; idx < warnings.length; idx++) {
            message = warnings[idx][1]
            rendered_text = getText(chart, message)
            show = warnings[idx][2]
            show_warning[message] |= show
        }

        for (var message in show_warning) {
            rendered_text = getText(chart, message)
            show = show_warning[message];

            if (show) {
                rendered_text.show()
            } else {
                rendered_text.hide()
            }
        }
    }


    function toggleAllSeriesWarnings(chart) {
        for (let idx = 0; idx < chart.series.length; idx++) {
            series = chart.series[idx]

            for (let idx = 0; idx < warnings.length; idx++) {
                let name = warnings[idx][0]
                if (series.name == name) {
                    warnings[idx][2] = !series.visible
                }
            }
        }

        updateChartWarnings(chart)
    }


    function toggleSeriesWarnings(series) {
        for (let idx = 0; idx < warnings.length; idx++) {
            let name = warnings[idx][0]
            if (series.name == name) {
                warnings[idx][2] = !series.visible
            }
        }

        updateChartWarnings(series.chart)
    }


    // This function is called for each individual chart. The instancedata contains the data for this chart.
    var result = function (instanceData) {
        //		A - x_value
        //		B - y_value
        //		C - flow_name
        //		D - height
        //		E - destination_name
        //		F - tick_label
        console.log("CCDF JAVASCRIPT : instanceData = " + instanceData)

        let latency_conversion = instanceData.latency_conversion;
        let time_unit = instanceData.parameter_time_unit

        if (typeof first_latency_distribution == 'undefined') { // making sure we do this block only once
            first_latency_distribution = instanceData.id
            latency_data = {
                title: {
                    text: null
                },
                chart: {
                    zoomType: 'xy',
                    panning: true,
                    panKey: 'shift',
                    type: 'line',
                    events: {
                        load: function () {
                            let chart = this;
                            let legend = chart.legend;
                            let move_up = 5;
                            let pos_x = legend.box.parentGroup.translateX;
                            let pos_y = legend.box.parentGroup.translateY - move_up;

                            let button = chart.renderer.button('Invert selection', pos_x, pos_y, function () {
                                toggleAllSeriesWarnings(chart)
                                Highcharts.each(chart.series, function (p, i) {
                                    if (p.visible) {
                                        p.setVisible(false, false);
                                    } else {
                                        p.setVisible(true, false);
                                    }
                                })
                                chart.redraw()
                            }, {
                                padding: 3,
                                width: 100,
                                'text-align': 'center'
                            })
                            button.add();

                            let button_width = button.box.parentGroup.width;
                            var title = legend.title;
                            title.translate(button_width + 3, -move_up);
                        }
                    }
                },

                tooltip: {
                    useHTML: true,
                    headerFormat: null,
                    pointFormatter: function () {
                        let y_value = this.y
                        if (y_value === 0.0000001) { // workaround to display logarithmic zero value
                            y_value = 0;
                        }

                        let ns_value = this.category.toFixed(3)
                        let p_value = (100 - y_value).toFixed(3)
                        let tooltiptext =
                            '<table><tr><td>Latency:</td><td><b>' + ns_value + ' ' + instanceData.parameter_time_unit + '</b></td>' +
                            '<tr><td style="color:' + this.series.color + '">' + this.series.name + ':</td><td><b>P' + p_value + '</b></td></tr>';

                        // add flow-specific warnings:
                        for (let idx = 0; idx < warnings.length; idx++) {
                            let flow_name = warnings[idx][0]
                            if (flow_name == this.series.name) {
                                let flow_warning = warnings[idx][1]

                                tooltiptext += '<tr><td colspan="2" style="color:' + excentis_orange + '">Warning: ' + flow_warning + '</td></tr>'
                            }
                        }

                        tooltiptext += '</table>'
                        return tooltiptext
                    },
                    valueDecimals: 2
                },
                xAxis: {
                    title: {
                        text: '<span style="font-family: \'DejaVu Sans\', Arial, Helvetica, sans-serif; color: #F7941C; font-size: 12px; line-height: 1.4640625; font-weight: bold;">Logarithmic Latency [' + instanceData.parameter_time_unit + ']</span><br>'
                    },
                    type: "logarithmic"
                },
                yAxis: [{
                    title: {
                        text: '<span style="font-family: \'DejaVu Sans\', Arial, Helvetica, sans-serif; color: #00A650; font-size: 12px; line-height: 1.2640625; font-weight: bold; ">Logarithmic CDF</span>'
                    },
                    opposite: true,
                    plotLines: [{
                        color: excentis_green,
                        width: 2,
                        dashStyle: 'dash',
                        value: 1 // draws a horizontal plotline at P99
                    }],
                    labels: {
                        formatter: function () {
                            let str = 'P';
                            str += (100.0 - this.value);
                            return str;
                        },
                        style: {
                            color: excentis_green
                        }
                    },
                    type: "logarithmic",
                    tickInterval: 1,
                    minorTickInterval: 0.1,
                    endOnTick: true,
                    gridLineWidth: 1,
                    max: 100.0,
                    min: 0.01
                }, {
                    title: {
                        text: '<span style="font-family: \'DejaVu Sans\', Arial, Helvetica, sans-serif; color: #00AEEF; font-size: 12px; line-height: 1.2640625; font-weight: bold; ">Logarithmic CCDF</span>'
                    },
                    labels: {
                        formatter: function () {
                            str = this.value / 100.0;
                            return str;
                        },
                        style: {
                            color: excentis_blue
                        }
                    },
                    type: "logarithmic",
                    tickInterval: 1,
                    minorTickInterval: 0.1,
                    endOnTick: true,
                    gridLineWidth: 1,
                    max: 100.0,
                    min: 0.01
                }
                ],
                plotOptions: {
                    series: {
                        marker: {
                            radius: 1,
                            symbol: "circle"
                        }, events: {
                            legendItemClick: function () {
                                toggleSeriesWarnings(this)
                            }
                        }
                    }
                },
                legend: {
                    title: {
                        style: '',// default is bold
                        text: 'or click on individual plots to show/hide:'
                    }
                },
                series: [],
                nanos: []
            }
        }

        // The lines directly below merge several chart elements into a single distribution chart.
        var plotLine = instanceData.series[0] // this will contain an entire plotline. The code is called multiple times while loading. Once for each plot line.
        if (typeof plotLine != 'undefined') { // when regenerating the entire graph, an emtpy series will be passed along. 
            var flowDestination = plotLine[0].C + " : " + plotLine[0].E;
            var drr = instanceData.parameter_data_rate_unit;
            var heights = [];
            let length = plotLine.length
            var distribution_map_nanos = [];

            // Four extra "fake" objects were added. One at the beginning and three at the end, 
            // to pass on additional data. (See GenerateLatencyCcdfCharts to understand)
            let nof_packets_below_min = plotLine[0].B //value_y_value see GenerateLatencyCcdfCharts why
            let total_nof_sampled_packets = Math.max(1, plotLine[length - 3].B) //value_y_value see GenerateLatencyCcdfCharts why
            let nof_packets_above_max = plotLine[length - 2].B //value_y_value see GenerateLatencyCcdfCharts why
            let total_nof_rx_packets = plotLine[length - 1].B //value_y_value see GenerateLatencyCcdfCharts why
            // and skip the last three objects from now on:
            length -= 3

			let sampled_total = nof_packets_below_min + total_nof_sampled_packets + nof_packets_above_max

            // start counting at the nof missed packets below range:
            let total_counted = nof_packets_below_min
            let used_buckets = 0
            // let bin_width = plotLine[2].A - plotLine[1].A 

            let use_linear_scale = false
            for (let idx = 0; idx < length && total_nof_sampled_packets > 0; idx++) {
                let element = plotLine[idx]
                value_x_nanos = element.A
                value_y_value = element.B

                if (idx == 0) { // reading the nof packets below min
                    continue // skip the nof packets below min
                }

                if (value_x_nanos != null && value_y_value != null) {

                    total_counted += value_y_value;
                    if (idx > 0 && value_y_value > 0) {
                        used_buckets++
                    }
                    //let center_of_bin = value_x_nanos + bin_width / 2.			
                    let normalized_y_value = 100 - (100 * total_counted / sampled_total);

                    // --- workaround to make the last value visible on log scale:
                    safe_x_value = above_zero(value_x_nanos)
                    safe_y_value = above_zero(normalized_y_value)

                    distribution_map_nanos.push([safe_x_value, safe_y_value])
                }
            }

            let serie = {
                name: flowDestination,
                nanos: distribution_map_nanos, // nanos is not a highcharts field, we added it so we can dynamically convert the latencies into the desired unit and put it in the data below: 
                data: [], // the actual data, in the desired unit, will be filled in below
                color: color_picker(),
                showInLegend: true
            }
            latency_data['series'].push(serie)


            let warning_index = 0;
            // Add warning:
            if (used_buckets < 50) {// --- There are 1000 buckets. If less than 50 were filled, maybe the user should use a different range in the Latency Project Properties ?
                warnings.push([flowDestination, warning_less_than_50_datapoints, true])
            }
            if (total_nof_sampled_packets < 10000) {// --- We decided on 22/4/2021 that 10.000 packets is a minimum to get a decent graph
                warnings.push([flowDestination, warning_less_than_10000_packets, true])
            }
            if (nof_packets_below_min > 0) {
                warnings.push([flowDestination, warning_packets_below_range, true])
            }
            if (nof_packets_above_max > 0) {
                warnings.push([flowDestination, warning_packets_above_range, true])
            }
            let expected_nof_samples = total_nof_rx_packets - nof_packets_below_min - nof_packets_above_max
            if (total_nof_sampled_packets < expected_nof_samples) {
                warnings.push([flowDestination, warning_packets_were_sampled, true])
            }
        } else {
            // Rerendering the chart, possibly with different latency unit:
            xAxis = latency_data['xAxis']
            xAxis.title.text = '<span style="font-family: \'DejaVu Sans\', Arial, Helvetica, sans-serif; color: #F7941C; font-size: 12px; line-height: 1.4640625; font-weight: bold;">Logarithmic Latency [' + instanceData.parameter_time_unit + ']</span><br>'
        }

        // Convert all latencies from nanos into the desired unit:
        let nof_plots = latency_data['series'].length
        for (let idx = 0; idx < nof_plots; idx++) {
            let serie = latency_data['series'][idx]
            let nof_dots = serie.nanos.length
            let converted_dots = []
            for (let dotid = 0; dotid < nof_dots; dotid++) {
                dot = serie.nanos[dotid]
                let x_nanos = dot[0]
                let y_value = dot[1]
                let x_in_desired_unit = latency_conversion(x_nanos)
                converted_dots.push([x_in_desired_unit, y_value])
            }
            serie.data = converted_dots
        }

        if (chart_layer != undefined) {
            cleanupChartWarnings(chart_layer)
        }

        chart_layer = Highcharts.chart(first_latency_distribution, latency_data);

        updateChartWarnings(chart_layer)

        //		Code to switch between LIN and LOG, currently disabled because it was sometimes slow.	      
        //		let btntext = 'Linear'
        //		chart.yAxisButton = chart.renderer.button(btntext, 647, 20)
        //		.attr({
        //            zIndex: 3
        //        })
        //        .on('click', function () {
        //			chart.xAxis[0].update({
        //				type: 'linear'
        //			})
        //			chart.yAxis[0].update({
        //				type: 'linear'
        //			})
        //			chart.yAxis[1].update({
        //				type: 'linear'
        //			})        	
        //        })
        //        .add();
        //		
        //		
        //		chart.renderer.button('Logarithmic', 707, 20)
        //		.attr({
        //            zIndex: 3
        //        })
        //        .on('click', function () {
        //			chart.xAxis[0].update({
        //				type: 'logarithmic'
        //			})
        //			chart.yAxis[0].update({
        //				type: 'logarithmic'
        //			})
        //			chart.yAxis[1].update({
        //				type: 'logarithmic'
        //			})
        //        })
        //        .add();
    }
    return result;
});