import pytest

import os
from typing import Union
import re

# FYI, Dockerfile requirements:
# > pip install selenium
# > pip install webdriver-manager

# selenium 4 imports:
import selenium
from selenium import webdriver

import logging

# Chromium:
from selenium.webdriver.chrome.service import Service as ChromiumService
from webdriver_manager.chrome import ChromeDriverManager

try:
    from webdriver_manager.core.utils import ChromeType
except ImportError:
    from webdriver_manager.core.os_manager import ChromeType

# Chrome:
from selenium.webdriver.chrome.service import Service as ChromeService
from webdriver_manager.chrome import ChromeDriverManager

# Edge:
from selenium.webdriver.edge.service import Service as EdgeService
from webdriver_manager.microsoft import EdgeChromiumDriverManager

# Firefox:
from selenium.webdriver.firefox.service import Service as FirefoxService
from webdriver_manager.firefox import GeckoDriverManager

# Extra imports used for testing:
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver.remote.webelement import WebElement

import pathlib
import json

# @pytest.fixture(scope='session')
# def report_files():
# Prepare list of JSON filenames
report_files = []

# Iterate directory
current_dir = os.getcwd()
for file_name in os.listdir(current_dir):
    logging.info(f'Looking for test reports in folder: {current_dir}')

    # Focus on only one specific JSON - the one with Run Title 'ONE TEST TO BIND THEM ALL'
    # if file_name.endswith('.json') and 'ONE TEST TO BIND THEM ALL' in file_name:
    if file_name.endswith('.json'):
        # The HTML page has the same filename as the JSON file, but .HTML instead of .JSON.
        # Just store the filename, without extension:
        file_name_base, ext = os.path.splitext(file_name)

        # Convert the file path to the URL format using the file:// protocol
        report_files.append(file_name_base)
    # return report_files


def make_headless(options):
    # Make the browser run headless, otherwise you'll get an error like:
    # 'Chrome failed to start: crashed...'
    options.add_argument('--headless')
    options.add_argument('--no-sandbox')
    options.add_argument('--disable-dev-shm-usage')
    return options

CHROME = 'chrome'

@pytest.fixture(scope='session')
def chrome():
    #Set up
    logging.info(f'Running Selenium version {selenium.__version__}')
    options = webdriver.ChromeOptions()
    options = make_headless(options)
    driver = webdriver.Chrome(service=ChromiumService(ChromeDriverManager(chrome_type=ChromeType.CHROMIUM).install()), options=options)
    yield driver
    #Teardown
    driver.quit()

CHROMIUM = 'chromium'

@pytest.fixture(scope='session')
def chromium():
    options = webdriver.ChromeOptions() 
    options = make_headless(options)
    driver = webdriver.Chrome(service=ChromiumService(ChromeDriverManager(chrome_type=ChromeType.CHROMIUM).install()), options=options)
    yield driver
    driver.quit()

FIREFOX = 'firefox'

@pytest.fixture(scope='session')
def firefox():
    options = webdriver.FirefoxOptions() 
    options = make_headless(options)
    driver = webdriver.Firefox(service=FirefoxService(GeckoDriverManager().install()), options=options)
    yield driver
    driver.quit()

EDGE = 'edge'

@pytest.fixture(scope='session')
def edge():
    options = webdriver.EdgeOptions()
    options = make_headless(options)
    driver = webdriver.Edge(service=EdgeService(EdgeChromiumDriverManager().install()), options=options)
    yield driver
    driver.quit()

# @pytest.fixture(scope='session')
# def all_drivers():
    # return [CHROME, CHROMIUM, EDGE, FIREFOX]
# all_drivers = [CHROME, CHROMIUM, EDGE, FIREFOX]
# Update 12/2/2024: Chrome and Chromium gave issues on Jenkins. We decided to disable them temporarily.
all_drivers = [EDGE, FIREFOX]

PAGE_LOAD_TIMEOUT = 5 # Seconds

# @pytest.fixture(scope='session')
# def bad_words():
#     return [
bad_words = [
    # 'DRAFT', # TODO REMOVE THIS ONE
    '?s', #if the microseconds symbol is not displayed right, it becomes '?s'
    'NaN',
    'Infinity',
    'NULL']


# Some words that should be in any report
# @pytest.fixture(scope='session')
# def good_words():
#     return [
good_words = [
    'Scenario Info',
    'Loss Legend',
    'Throughput Legend',
    'Throughput Units',
    'Latency Units',
    'ByteBlower Ports']


def read_json_file(file_path):
    with open(file_path) as json_file:
        data = json.load(json_file)
    return data

# Translates some JSON into HTML
# If the specified value is None, this means that the value found in the JSON must be used instead.
keys_to_check = {
    'scenarioName': None,
    'projectAuthor': None,
    'runTitle': None,
    # 'startMoment': None,
    # 'endMoment': None,
    'scenarioState': None,
    'projectName': None,
    'guiVersion': None,
    'ipv4Ports': 'IPv4 ByteBlower Ports',
    'ipv6Ports': 'IPv6 ByteBlower Ports',
    'rfc2544Flows': 'RFC-2544 Throughput Benchmark'
    # 'frameBlastingFlows': ['Frame Blasting Flows: Info','Frame Blasting Flows: Throughput']
}


# def add_check(key, value, json_string, html_string):
#     if key in json_string and value:
#         good_words.append(html_string)
#     else:
#         bad_words.append(html_string)


def update_things_to_check(key, value):
    if key in keys_to_check and value:
        to_check = keys_to_check[key]
        if to_check:
            return to_check
        else:
            return value
    return None

# Check every second if the page is already loaded.
# Until the timeout is reached.
def verify_page_is_loaded_quickly(driver):
    for second in range(1, PAGE_LOAD_TIMEOUT):
        driver.implicitly_wait(1)
        state = driver.execute_script("return document.readyState")
        if state == "complete":
            return second
    return None


def load_page(driver, url):
    driver.get(url)
    load_time = verify_page_is_loaded_quickly(driver)
    assert load_time, f'It took longer than {PAGE_LOAD_TIMEOUT}s to load.'
    logging.info(f'The ByteBlower report loaded in {load_time}s')


def parse_json(json_file_path):
    json_data = read_json_file(json_file_path)
    extra_good_words = []
    for key, value in json_data.items():
        extra_html_word = update_things_to_check(key, value)
        if extra_html_word:
            extra_good_words.append(extra_html_word)
    return extra_good_words


def initialize(request, driver_type, report_file):
    # Converting browser identifier string into browser driver:
    # logging.info(f'Instantiating driver for {driver_type}')
    driver: Union[webdriver.Chrome, webdriver.Edge, webdriver.Firefox] = request.getfixturevalue(driver_type)

    # Get the absolute path to the local JSON and HTML file.
    # The HTML file is the report generated using the data inside the JSON file.
    json_file_path = os.path.abspath(report_file + '.json')
    html_file_path = os.path.abspath(report_file + '.html')

    assert os.path.exists(json_file_path), f'JSON file not found'
    assert os.path.exists(html_file_path), f'HTML file not found'

    url = pathlib.Path(html_file_path).as_uri()

    loaded_url = driver.current_url
    # Only load the page once:
    if loaded_url != url:
        load_page(driver, url)

    return driver


# skipping Chromium because it is very similar to Chrome which is being tested.
# Skipping Edge because it doesn't work
@pytest.mark.parametrize('driver_type', all_drivers)
@pytest.mark.parametrize('report_file', report_files)
def test_title(request, driver_type: str, report_file: str):
    driver = initialize(request, driver_type, report_file)

    # Specify the expected title you want to check
    expected_title = 'ByteBlower Report'

    title_text = driver.title
    # logging.info(f'Title Text is {title_text}')

    # Check if the expected title is present in the page
    assert expected_title in title_text


def get_visible_text(driver: Union[webdriver.Chrome, webdriver.Edge, webdriver.Firefox]):
    try:
        # Get root element:
        element:WebElement = driver.find_element(By.XPATH, "//*")

        # Loop through each element and get its visible text
        # for element in elements:
        if element.is_displayed():
            # xpath = element.get_attribute("xpath")
            # print(xpath)
            return element
    except NoSuchElementException:
        assert False, f'No elements were found!'


@pytest.mark.parametrize('driver_type', all_drivers)
@pytest.mark.parametrize('report_file', report_files)
# @pytest.mark.parametrize('bad_word', bad_words)
def test_look_for_bad_words(request, driver_type: str, report_file: str):
    driver = initialize(request, driver_type, report_file)
    root_element = get_visible_text(driver)
    text = root_element.text

    assert text, f'Root element text is empty.'

    # print(text)
    # Check if the word is present in the visible text:
    # found_bad_word = re.search(bad_word, text, re.IGNORECASE)
    for bad_word in bad_words:
        logging.info(f'Verifying that the following word is NOT present in the HTML report: {bad_word}')
        print(f'Verifying that the following word is NOT present in the HTML report: {bad_word}')
        found_bad_word = bad_word in text
        if found_bad_word:
            # Capture a screenshot of the WebElement:
            screenshot_filename = f'found_bad_word__{bad_word}__{driver_type}__{report_file}.png'
            screenshot_filename = screenshot_filename.replace('?', "QUESTIONMARK_instead_of_micro_" )
            root_element.screenshot(screenshot_filename)
        assert not found_bad_word, f'Found bad word: >>>>{bad_word}<<<<. Screenshot was taken: {screenshot_filename}'


@pytest.mark.parametrize('driver_type', all_drivers)
@pytest.mark.parametrize('report_file', report_files)
# @pytest.mark.parametrize('good_word', good_words)
def test_look_for_good_words(request, driver_type: str, report_file: str):
    driver = initialize(request, driver_type, report_file)
    root_element = get_visible_text(driver)
    text = root_element.text

    # print(text)
    # logging.info(f'Verifying that the following words are present in the HTML report: {good_words}')

    json_file_path = os.path.abspath(report_file + '.json')
    extra_good_words = parse_json(json_file_path)
    all_good = good_words + extra_good_words
    for good_word in all_good:
        # Check if the word is present in the visible text:
        logging.info(f'Verifying that the following word is present in the HTML report: {good_word}')
        print(f'Verifying that the following word is present in the HTML report: {good_word}')

        found_good_word = good_word in text
        if not found_good_word:
            # Capture a screenshot of the WebElement:
            screenshot_filename = f'good_word_missing__{good_word}_{driver_type}_{report_file}.png'
            screenshot_filename = screenshot_filename.replace('?', "QUESTIONMARK_instead_of_micro_" )
            root_element.screenshot(screenshot_filename)
        assert found_good_word, f'Failed to find good word: <<<<{good_word}>>>>. Screenshot was taken: {screenshot_filename}'


# Currently just clicking on the item.
# If that causes any javascript error, 
# it will be logged and that will cause a fail.
def click_on_element(driver, element_id):
    try:
        element = driver.find_element('id', element_id)
        element.click()
    except NoSuchElementException:
        assert False, f'Element was not found: {element_id}'


@pytest.mark.parametrize('driver_type', all_drivers)
@pytest.mark.parametrize('report_file', report_files)
def test_throughput_units(request, driver_type: str, report_file: str):
    driver = initialize(request, driver_type, report_file)
    click_on_element(driver, 'inlineRadio_bps')
    click_on_element(driver, 'inlineRadio_kbps')
    click_on_element(driver, 'inlineRadio_Mbps')
    click_on_element(driver, 'inlineRadio_Gbps')


@pytest.mark.parametrize('driver_type', all_drivers)
@pytest.mark.parametrize('report_file', report_files)
def test_latency_units(request, driver_type: str, report_file: str):
    driver = initialize(request, driver_type, report_file)
    click_on_element(driver, 'inlineRadio_NANO')
    click_on_element(driver, 'inlineRadio_MICRO')
    click_on_element(driver, 'inlineRadio_MILLI')


#driver.get_log is not implemented by Firefox driver.
@pytest.mark.parametrize('driver_type', [EDGE])
@pytest.mark.parametrize('report_file', report_files)
def test_no_javascript_errors(request, driver_type: str, report_file: str):
    driver = initialize(request, driver_type, report_file)

    # Retrieve the JavaScript error logs from the browser
    logs = driver.get_log('browser')

    if len(logs) > 0:
        logging.info('JavaScript log:')
        
    for log in logs:
        logging.info(log['message'])

    # Check if any JavaScript errors were logged
    error_logs = [log for log in logs if log['level'] == 'SEVERE']
    assert len(error_logs) == 0, f'Found {len(error_logs)} JavaScript error(s)'
    
