import pytest

import os
from typing import Union
import re

# FYI, Dockerfile requirements:
# > pip install selenium
# > pip install webdriver-manager
# > pip install python-dateutil

# selenium 4 imports:
import selenium
from selenium import webdriver

import logging

from dateutil import parser

# Chromium:
from selenium.webdriver.chrome.service import Service as ChromiumService
from webdriver_manager.chrome import ChromeDriverManager

try:
    from webdriver_manager.core.utils import ChromeType
except ImportError:
    from webdriver_manager.core.os_manager import ChromeType

# Chrome:
from selenium.webdriver.chrome.service import Service as ChromeService
from webdriver_manager.chrome import ChromeDriverManager

# Edge:
from selenium.webdriver.edge.service import Service as EdgeService
from webdriver_manager.microsoft import EdgeChromiumDriverManager

# Firefox:
from selenium.webdriver.firefox.service import Service as FirefoxService
from webdriver_manager.firefox import GeckoDriverManager

# Extra imports used for testing:
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver.remote.webelement import WebElement

import pathlib
import json

headless = True

# Prepare list of JSON filenames
report_files = []

def read_json_file(file_path):
    with open(file_path) as json_file:
        data = json.load(json_file)
    return data

# Iterate directory
current_dir = os.getcwd()
for file_name in os.listdir(current_dir):
    logging.info(f'Looking for test reports in folder: {current_dir}')

    if file_name.startswith('basic - '):
        if file_name.endswith('.json'):
            json_file = file_name
            # Get data out of the original JSON file
            json_file_path = os.path.abspath(json_file)
            assert os.path.exists(json_file_path), f'JSON file not found'
            json_data = read_json_file(json_file_path)
            rx = json_data['frameBlastingFlows'][0]['destinations'][0]['received']
        if file_name.endswith('.html'):
            #if file_name.endswith('.html') and 'basic - ' in file_name:
            # The HTML page has the same filename as the JSON file, but .HTML instead of .JSON.
            # Just store the filename, without extension:
            file_name_base, ext = os.path.splitext(file_name)
            report_files.append(file_name_base)

def make_headless(options):
    # Make the browser run headless, otherwise you'll get an error like:
    # 'Chrome failed to start: crashed...'
    options.add_argument('--headless')
    options.add_argument('--no-sandbox')
    options.add_argument('--disable-dev-shm-usage')
    return options

CHROME = 'chrome'

@pytest.fixture(scope='session')
def chrome():
    #Set up
    logging.info(f'Running Selenium version {selenium.__version__}')
    options = webdriver.ChromeOptions()
    if headless:
        options = make_headless(options)
    driver = webdriver.Chrome(service=ChromiumService(ChromeDriverManager(chrome_type=ChromeType.CHROMIUM).install()), options=options)
    yield driver
    #Teardown
    driver.quit()

CHROMIUM = 'chromium'

@pytest.fixture(scope='session')
def chromium():
    options = webdriver.ChromeOptions() 
    if headless:
        options = make_headless(options)
    driver = webdriver.Chrome(service=ChromiumService(ChromeDriverManager(chrome_type=ChromeType.CHROMIUM).install()), options=options)
    yield driver
    driver.quit()

FIREFOX = 'firefox'

@pytest.fixture(scope='session')
def firefox():
    options = webdriver.FirefoxOptions() 
    if headless:
        options = make_headless(options)
    driver = webdriver.Firefox(service=FirefoxService(GeckoDriverManager().install()), options=options)
    yield driver
    driver.quit()

EDGE = 'edge'

@pytest.fixture(scope='session')
def edge():
    options = webdriver.EdgeOptions()
    options = make_headless(options)
    driver = webdriver.Edge(service=EdgeService(EdgeChromiumDriverManager().install()), options=options)
    yield driver
    driver.quit()

all_drivers = [CHROME, CHROMIUM, EDGE, FIREFOX]
# Dropping Chrome and Chromium for now.
# These don't work nicely with testing inside a Docker container.
# (various unrelated crashes)
all_drivers = [EDGE, FIREFOX]


def initialize(request, driver_type, report_file):
    # Converting browser identifier string into browser driver:
    # logging.info(f'Instantiating driver for {driver_type}')
    driver: Union[webdriver.Chrome, webdriver.Edge, webdriver.Firefox] = request.getfixturevalue(driver_type)

    # Get the absolute path to the local JSON and HTML file.
    # The HTML file is the report generated using the data inside the JSON file.
    html_file_path = os.path.abspath(report_file + '.html')

    assert os.path.exists(html_file_path), f'HTML file not found'

    url = pathlib.Path(html_file_path).as_uri()

    loaded_url = driver.current_url
    # Only load the page once:
    if loaded_url != url:
        driver.get(url)

    return driver

@pytest.fixture(scope='session', autouse=True)
def rx_bytes(request):
    return rx['bytes']

@pytest.fixture(scope='session', autouse=True)
def rx_packets(request):
    return rx['packets']

@pytest.fixture(scope='session', autouse=True)
def rx_duration(request):
    rx_firstPacketTime = parser.parse(rx['firstPacketTime'])
    rx_lastPacketTime = parser.parse(rx['lastPacketTime'])
    duration = rx_lastPacketTime - rx_firstPacketTime
    rx_seconds = duration.total_seconds()
    return rx_seconds

# skipping Chromium because it is very similar to Chrome which is being tested.
# Skipping Edge because it doesn't work
# This test verifies that the displayed throughput is calculated using the desired option 
# specifying which bytes need to be counted for each frame: FRAME_ONLY / FRAME_AND_FCS / FRAME_AND_ALL_OTHER_FIELDS
@pytest.mark.parametrize('driver_type', all_drivers)
@pytest.mark.parametrize('report_file', report_files)
def test_throughput_calculated_correctly(request, driver_type: str, report_file: str, rx_bytes, rx_packets, rx_duration):
    driver = initialize(request, driver_type, report_file)

    # Make sure that the throughput is displayed in bps, 
    # to get as much detail as possible
    click_on_element(driver, 'inlineRadio_bps')

    # Locate the Average Throughput cell 
    # within the 'Frame Blasting Flows: Throughput' table 
    throughput_cell_locator_xpath = "//html//body//div[1]//table[3]//tbody//tr[2]//td[11]"
    cell = driver.find_element(By.XPATH, throughput_cell_locator_xpath)
    throughput = cell.text

    # remove commas:
    html_throughput = throughput.replace(',', '')
    reported_throughput = float(html_throughput)

    extra_bytes = 0
    if 'FRAME_ONLY' in report_file:
        extra_bytes = 0 
    elif 'FRAME_AND_FCS' in report_file:
        extra_bytes = 4
    elif 'FRAME_AND_ALL_OTHER_FIELDS' in report_file:
        extra_bytes = 24
    else:
        return

    extra_bytes = extra_bytes * rx_packets
    expected_throughput = (rx_bytes + extra_bytes) * 8 / rx_duration

    assert expected_throughput == pytest.approx(reported_throughput)

# Currently just clicking on the item.
# If that causes any javascript error, 
# it will be logged and that will cause a fail.
def click_on_element(driver, element_id):
    try:
        element = driver.find_element('id', element_id)
        element.click()
    except NoSuchElementException:
        assert False, f'Element was not found: {element_id}'
